/**
 * @file gain_mode.hpp
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief Gain mode
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include "gain_mode.h"

namespace brainaccess::core
{
	/**
	 * @brief Gain mode enum
	 */
	enum class gain_mode : ba_gain_mode
	{
		X1 = BA_GAIN_MODE_X1,          ///< 1x gain
		X2 = BA_GAIN_MODE_X2,          ///< 2x gain
		X4 = BA_GAIN_MODE_X4,          ///< 4x gain
		X6 = BA_GAIN_MODE_X6,          ///< 6x gain
		X8 = BA_GAIN_MODE_X8,          ///< 8x gain
		X12 = BA_GAIN_MODE_X12,        ///< 12x gain
		X24 = BA_GAIN_MODE_X24,        ///< 24x gain
		UNKNOWN = BA_GAIN_MODE_UNKNOWN ///< Unknown gain
	};

	/**
	 * @brief Converts gain mode to integer multiplier representing the gain mode
	 * (ex: X12 returns 12) 
	 * 
	 * @param g Gain mode to convert to multiplier
	 * @return Integer multiplier representing the gain mode (ex: X12 returns 12) 
	 */
	[[maybe_unused]] [[nodiscard]] static int gain_mode_to_multiplier(const gain_mode g) noexcept
	{
		return ba_gain_mode_to_multiplier(static_cast<ba_gain_mode>(g));
	}
	
	/**
	 * @brief Attempts to convert multiplier to gain mode (ex: 12 returns X12)
	 * 
	 * @param g Multiplier to convert to gain mode
	 * @return Gain mode
	 */
	[[maybe_unused]] [[nodiscard]] static gain_mode multiplier_to_gain_mode(const int g) noexcept
	{
		return static_cast<gain_mode>(ba_multiplier_to_gain_mode(g));
	}
}
