/**
 * @file eeg_manager.hpp
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief EEG device manager
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include "eeg_manager.h"
#include "callbacks.hpp"
#include "device_info.hpp"
#include "eeg_channel.hpp"
#include "annotation.hpp"
#include "battery_info.hpp"
#include "full_battery_info.hpp"
#include "impedance_measurement_mode.hpp"
#include "gain_mode.hpp"
#include "error.hpp"
#include "polarity.hpp"

namespace brainaccess::core
{
	inline namespace abi_0
	{
		/**
		 * @brief The EEG manager is the primary tool for communicating with the
		 * BrainAccess device. Note that the EEG manager is not thread-safe.
		 */
		class BA_CORE_DLL_EXPORT eeg_manager
		{
		public:
			/**
			 * @brief Destroys an EEG manager instance.
			 * 
			 * @details be called exactly once, after the manager is no longer needed
			 */
			virtual ~eeg_manager() noexcept = default;

			/**
			 * @brief Connects to a device via COM port and attempts to initialize it.
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * You must wait for the callback to complete before doing anything else with
			 * the EEG manager. The boolean parameter of the callback is true if the
			 * connection is successful, false otherwise.
			 * 
			 * @param port `COMx` on Windows, `/dev/rfcommX` on Linux
			 * @param callback Function to be called after the connection succeeds
			 * @param data Data to be passed to the callback
			 */
			virtual void connect(const char* port, ba_callback_future_bool callback, void* data) noexcept = 0;
			
			/**
			 * @brief Disconnects the EEG manager from the device, if connected.
			 */
			virtual void disconnect() noexcept = 0;
			
			/**
			 * @brief Checks if an EEG manager is connected to any device
			 * 
			 * @return `true` if EEG manager is connected
			 */
			[[nodiscard]] virtual bool is_connected() noexcept = 0;

			/**
			 * @brief Starts streaming data from the device
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * You must not call this function twice without stopping the stream in between.
			 * 
			 * @param callback Function to be called after the stream starts
			 * @param data Data to be passed to the callback
			 * @return Error code
			 */
			virtual error start_stream(ba_callback_future_void callback, void* data) noexcept = 0;
			
			/**
			 * @brief Stops streaming data from the device
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * You must not call this function twice without starting the stream in between.
			 * You must not call this function while the stream is not running.
			 * 
			 * Calling this function resets all stream settings. If you want to stream again
			 * afterwards, you must re-enable all the channels, biases, gains, and impedance
			 * measurement mode that you set previously.
			 * 
			 * @param callback Function to be called after the stream stops
			 * @param data Data to be passed to the callback
			 * @return Error code 
			 */
			virtual error stop_stream(ba_callback_future_void callback, void* data) noexcept = 0;
			
			/**
			 * @brief Checks if the device is streaming
			 * 
			 * @return `true` if the device is currently streaming
			 */
			[[nodiscard]] virtual bool is_streaming() const noexcept = 0;

			/**
			 * @brief 
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * The digital input pin, which by default is pulled high but can be pulled low
			 * by an external sensor, can also be pulled low by the device itself. By
			 * default, upon powering up or connecting/disconnecting the device, the digital
			 * input pin is pulled high.
			 * 
			 * This can be useful, for example, in case you want to synchronize devices:
			 * connect device A and B's digital inputs, start both streams, then set A's
			 * digital input to pull low, which also pulls B's input with it. The falling
			 * edge can be recorded from both streams, and the data can then be aligned
			 * accordingly.
			 * 
			 * This can also be used for low-speed communication with external devices,
			 * controlling LEDs via a mosfet, etc.
			 * 
			 * @param pin Number of digital input pin of the EEG device to set the IO state
			 * of (starting from 0)
			 * @param state `true` to pull high, `false` to pull to ground
			 * @param callback Function to be called after the stream stops
			 * @param data Data to be passed to the callback
			 * @return Error code
			 */
			virtual error set_io(uint8_t pin, bool state, ba_callback_future_void callback, void* data) noexcept = 0;

			/**
			 * @brief Gets the standard battery info from an EEG device
			 * 
			 * @details The device periodically sends its battery info after an update.
			 * That info is stored in a cache and can be retrieved with this function.
			 * 
			 * @return Cached battery info
			 */
			[[nodiscard]] virtual battery_info get_battery_info() noexcept = 0;
			
			/**
			 * @brief Gets the extended battery info from an EEG device
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * Sends a request to the device for the full battery info. No caching is
			 * performed on this data.
			 * 
			 * @param callback Function to be called after the info is received
			 * @param data Data to be passed to the callback
			 * @return Error code
			 */
			virtual error get_full_battery_info(ba_callback_future_full_battery_info callback, void* data) noexcept = 0;
			
			/**
			 * @brief Measure approximate communication latency with the device
			 * 
			 * @details This function runs asynchronously. This means that this function
			 * does not block, and in order to get the result, you must use the callback.
			 * The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * Performs a single ping request and measures the time in seconds. The float
			 * parameter of the callback is set to the latency.
			 * 
			 * @param callback Function to be called after the latency is measured
			 * @param data Data to be passed to the callback
			 * @return Error code
			 */
			virtual error get_latency(ba_callback_future_float callback, void* data) noexcept = 0;

			/**
			 * @brief Enables the channel on the device and adds the data to the stream
			 * chunks
			 * 
			 * @details This function takes effect on stream start, and its effects are
			 * reset by stream stop. Therefore, it must be called with the appropriate
			 * arguments before every stream start.
			 * 
			 * @param ch Channel ID of the channel to enable/disable
			 * @param state True for enable, false for disable
			 */
			virtual void set_channel_enabled(eeg_channel ch, bool state) noexcept = 0;
			
			/**
			 * @brief Changes gain mode for a channel on the device.
			 * 
			 * @details This function takes effect on stream start, and its effects are
			 * reset by stream stop. Therefore, it must be called with the appropriate
			 * arguments before every stream start.
			 * 
			 * This only affects channels that support it. For example, it affects the
			 * electrode measurement channels but not sample number or digital input.
			 * 
			 * Note that the affected channel data is already multiplied by the gain you set
			 * here.
			 * 
			 * @param ch Channel ID of the channel to modify the gain of
			 * @param g Gain mode
			 */
			virtual void set_channel_gain(eeg_channel ch, gain_mode g) noexcept = 0;
			
			/**
			 * @brief Set an electrode channel as a bias electrode
			 * 
			 * @details This function takes effect on stream start, and its effects are
			 * reset by stream stop. Therefore, it must be called with the appropriate
			 * arguments before every stream start.
			 * 
			 * This only affects channels that support it. For example, it affects the
			 * electrode measurement channels but not sample number or digital input.
			 * 
			 * Sets channel that is used for bias feedback. The set channel signal is used
			 * to drive the bias electrode and cable shields to actively cancel common mode 
			 * noise such as noise from the mains. Please select channel that is believed to
			 * have a relatively good signal.
			 * 
			 * @param ch Channel ID of the electrode to set as bias
			 * @param p Which side of the electrode to use (if device is not
			 * bipolar, use BOTH)
			 */
			virtual void set_channel_bias(eeg_channel ch, polarity p) noexcept = 0;
			
			/**
			 * @brief Sets impedance measurement mode
			 * 
			 * @details This function takes effect on stream start, and its effects are
			 * reset by stream stop. Therefore, it must be called with the appropriate
			 * arguments before every stream start.
			 * 
			 * This function setups device for electrode impedance measurement. It injects a
			 * 7nA certain frequency current through the bias electrodes to measurement
			 * electrodes. Voltage recordings from each channel can then be used to
			 * calculate the impedance for each electrode: Impedance = Vpp/7nA
			 * 
			 * @param mode Impedance mode to set
			 */
			virtual void set_impedance_mode(impedance_measurement_mode mode) noexcept = 0;

			/**
			 * @brief Get EEG device info
			 * 
			 * @details Must not be called unless device connection is successful.
			 * 
			 * @return Info of the currently connected device
			 */
			[[nodiscard]] virtual const device_info& get_device_info() const noexcept = 0;

			/**
			 * @brief Gets the index of a channel's data into the chunk
			 * 
			 * @details Must only be used during stream (after stream start, before stream
			 * stop).
			 * 
			 * A chunk is an array of channel data (channel data being an array of values).
			 * To get the index for the channel you're looking for, use this function.
			 * 
			 * If channel was not enabled, returns (size_t)-1
			 * 
			 * @param ch Channel to get the index of
			 * @return Index into chunk representing a channel
			 */
			[[nodiscard]] virtual size_t get_channel_index(eeg_channel ch) const noexcept = 0;

			/**
			 * @brief Gets device sample rate
			 * 
			 * @return Sample frequency (Hz)
			 */
			[[nodiscard]] virtual uint16_t get_sample_frequency() const noexcept = 0;

			/**
			 * @brief Sets a callback to be called every time a chunk is available
			 * 
			 * @details The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * Set to null to disable.
			 * 
			 * @param callback Function to be called every time a chunk is available
			 * @param data Data to be passed to the callback
			 */
			virtual void set_callback_chunk(ba_callback_chunk callback, void* data) noexcept = 0;
			
			/**
			 * @brief Sets a callback to be called every time the battery status is updated
			 * 
			 * @details The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * Set to null to disable.
			 * 
			 * @param callback Function to be called every time a battery update is available
			 * @param data Data to be passed to the callback
			 */
			virtual void set_callback_battery(ba_callback_battery callback, void* data) noexcept = 0;
			
			/**
			 * @brief Sets a callback to be called every time the device disconnects
			 * 
			 * @details The callback may or may not run in the reader thread, and as such,
			 * synchronization must be used to avoid race conditions, and the callback
			 * itself must be as short as possible to avoid blocking communication with the
			 * device.
			 * 
			 * Set to null to disable.
			 * 
			 * @param callback Function to be called every time the device disconnects
			 * @param data Data to be passed to the callback
			 */
			virtual void set_callback_disconnect(ba_callback_disconnect callback, void* data) noexcept = 0;

			/**
			 * @brief Add an annotation with the current timestamp and given string
			 * 
			 * @details Should not be used before stream start.
			 * 
			 * Note that annotations are cleared on disconnect.
			 * 
			 * @param annotation Annotation string to add
			 */
			virtual void annotate(const char* annotation) noexcept = 0;
			
			/**
			 * @brief Retrieve all the annotations accumulated so far
			 * 
			 * @details Note that annotations are cleared on disconnect
			 * 
			 * @param annotations (Output parameter) Annotation array
			 * @param annotations_size (Output parameter) Annotation array size
			 */
			virtual void get_annotations(annotation** annotations, size_t* annotations_size) const noexcept = 0;
			
			/**
			 * @brief Clears all accumulated annotations
			 * 
			 * @details Note that annotations are cleared on disconnect
			 */
			virtual void clear_annotations() noexcept = 0;
		};
	}

	/**
	 * @brief Creates a new EEG manager instance
	 * 
	 * @return EEG manager instance handle
	 */
	[[maybe_unused]] [[nodiscard]] static eeg_manager* eeg_manager_new() noexcept
	{
		return static_cast<eeg_manager*>(ba_eeg_manager_new());
	}
}
