/**
 * @file eeg_channel.hpp
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief EEG data stream channel ID constants
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include "eeg_channel.h"

namespace brainaccess::core
{
	typedef ba_eeg_channel eeg_channel;

	namespace eeg_channel_id
	{
		/**
		 * @brief The number of the sample starting from 0 at stream start
		 * 
		 * @details Data type: `size_t`
		 */
		static constexpr eeg_channel SAMPLE_NUMBER = BA_EEG_CHANNEL_ID_SAMPLE_NUMBER;
		
		/**
		 * @brief EEG electrode measurement value (uV)
		 * 
		 * @details Data type: `double`
		 */
		static constexpr eeg_channel ELECTRODE_MEASUREMENT = BA_EEG_CHANNEL_ID_ELECTRODE_MEASUREMENT;
		
		/**
		 * @brief Whether or not the electrode is making contact with the skin
		 * 
		 * @details Data type: `bool`
		 * 
		 * If the device has separate P and N electrodes, the value will be
		 * true only if both electrodes are making contact.
		 */
		static constexpr eeg_channel ELECTRODE_CONTACT = BA_EEG_CHANNEL_ID_ELECTRODE_CONTACT;
		
		/**
		 * @brief Whether or not the positive (P) electrode is making contact
		 * with the skin
		 * 
		 * @details Data type: `bool`
		 */
		static constexpr eeg_channel ELECTRODE_CONTACT_P = BA_EEG_CHANNEL_ID_ELECTRODE_CONTACT_P;

		/**
		 * @brief Whether or not the negative (N) electrode is making contact
		 * with the skin
		 * 
		 * @details Data type: `bool`
		 */
		static constexpr eeg_channel ELECTRODE_CONTACT_N = BA_EEG_CHANNEL_ID_ELECTRODE_CONTACT_N;

		/**
		 * @brief Digital IO pin state
		 * 
		 * @details Data type: `bool`
		 */
		static constexpr eeg_channel DIGITAL_INPUT = BA_EEG_CHANNEL_ID_DIGITAL_INPUT;
		
		/**
		 * @brief Accelerometer value
		 * 
		 * @details Data type: `float`
		 */
		static constexpr eeg_channel ACCELEROMETER = BA_EEG_CHANNEL_ID_ACCELEROMETER;

		/**
		 * @brief Gyroscope value
		 * 
		 * @details Data type: `float`
		 */
		static constexpr eeg_channel GYROSCOPE = BA_EEG_CHANNEL_ID_GYROSCOPE;
	}
}
