/**
 * @file device_features.hpp
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief Device feature query functionality
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include "device_info.hpp"
#include "device_features.h"

namespace brainaccess::core
{
	inline namespace abi_0
	{
		/**
		 * @brief Base class allowing the user to check what features a particular
		 * device supports
		 */
		class BA_CORE_DLL_EXPORT device_features
		{
		public:
			virtual ~device_features() noexcept = default;

			/**
			 * @brief Whether or not the device can capture gyroscope data
			 * 
			 * @return true if device has a gyroscope, false otherwise
			 */
			[[nodiscard]] virtual bool has_gyro() const noexcept = 0;

			/**
			 * @brief Whether or not the device's electrodes are bipolar
			 * 
			 * @details Bipolar electrodes have separate P (positive) and N
			 * (negative) contacts
			 * 
			 * @return true if electrodes are bipolar, false otherwise
			 */
			[[nodiscard]] virtual bool is_bipolar() const noexcept = 0;

			/**
			 * @brief Gets the number of EEG/EMG electrodes supported by the device
			 * 
			 * @return Number of electrodes
			 */
			[[nodiscard]] virtual uint8_t electrode_count() const noexcept = 0;
		};
	}

	/**
	 * @brief Gets pointer to device_features instance
	 * 
	 * @details The pointer is guaranteed to be statically allocated, so no need
	 * to delete or do any kind of memory management.
	 * 
	 * @param i Device for which to get features. Serial number is ignored.
	 * @return Pointer to corresponding device_features instance, or nullptr if
	 * the device is not supported
	 */
	[[maybe_unused]] [[nodiscard]] static const device_features* device_features_get(const device_info& i) noexcept
	{
		return static_cast<const device_features*>(ba_core_device_features_get(&i));
	}
}
