/**
 * @file bacore.h
 * @author Neurotechnology (brainaccess@neurotechnology.com)
 * @brief Core library info and lifecycle API
 * 
 * @copyright Copyright (c) 2022 Neurotechnology
 */

#pragma once

#include <stdint.h>
#include "version.h"
#include "dllexport.h"

/**
 * @brief Unknown error while calling `ba_core_init()`
 */
#define BA_INIT_ERROR_UNKNOWN 0xFF

/**
 * @brief Initialized successfully
 */
#define BA_INIT_ERROR_OK 0

/**
 * @brief Configuration file contains a value of the wrong type
 */
#define BA_INIT_ERROR_CONFIG_TYPE 1

/**
 * @brief Configuration file could not be parsed, invalid JSON or missing value
 */
#define BA_INIT_ERROR_CONFIG_PARSE 2

/**
 * @brief The version requested and the version 
 */
#define BA_INIT_ERROR_INCOMPATIBLE_VERSION 3

/**
 * @brief Error codes for init
 */
typedef uint8_t ba_init_error;

#ifdef __cplusplus
extern "C"
{
#endif //__cplusplus

/**
 * @brief Initializes the library. Must be called before any other BrainAccess
 * Core library function. Only call once.
 * 
 * @details This function reads the config file, starts logging, etc. It first
 * checks if the version of the library that the application expects and the
 * version of the library installed are compatible.
 * 
 * @param version The version of the library that the application expects.
 * @return Error code
 */
BA_CORE_DLL_EXPORT ba_init_error ba_core_init(const ba_version* version) NOEXCEPT;

/**
 * @brief Returns the installed library's actual version
 * @details Uses semantic versioning
 * 
 * @return Installed library's actual version
 */
BA_CORE_DLL_EXPORT const ba_version* ba_core_get_version() NOEXCEPT;

/**
 * @brief Closes the library and cleans up afterwards. Must be called after all
 * BrainAccess Core libary functions used by the application. Only call once.
 * 
 * @details If `ba_core_init()` failed, do not call this function.
 */
BA_CORE_DLL_EXPORT void ba_core_close() NOEXCEPT;

#ifdef __cplusplus
}
#endif //__cplusplus
